//******************************************************************************
//   MSP-FET430P140 Demo - SMBus Master reads from SMBus MSP430 Slave with PEC
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. The master 
//   reads from the slave. This is the master code. The slave code is called
//   fet140_SMB_slave_PEC.c. The data from the slave transmitter begins
//   at 1 and is incremented with each transfer.The RXRDYIFG interrupt is used 
//   to know when new data has been received.
//   A CRC-8 PEC byte is appended after each data byte by the slave
//   and is validated by the master. If not correct LED P1.0 is set.
//   ACLK = n/a, MCLK = SMCLK = I2CIN = DCO ~ 800kHz
//   //* MSP430F169 Device Required *//
//   
//                                 /|\  /|\
//                  MSP430F169     10k  10k     MSP430F169
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1             |
//             |                 |  |      |             P1.0|-->LED
//             |                 |  |      |                 |
//             |             P3.3|<-+----->|P3.3             |
//             |                 |         |                 |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  Feb 2005
//  Built with IAR Embedded Workbench Version: 3.21A
//******************************************************************************

#include  <msp430x16x.h>
#define CRC_MSG_SIZE	   4
#define CRC8_POLY	   0x07
#define CRC8_INIT_REM      0x0

unsigned char crc8MakeBitwise(unsigned char crc, unsigned char poly, unsigned char *pmsg, unsigned int msg_size);

unsigned int length;
unsigned char crc8;
unsigned char msg[CRC_MSG_SIZE] = {0};
unsigned char RXData = 0;
unsigned char TestData = 0;

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;             // Stop WDT
  P3SEL |= 0x0A;                        // Select I2C pins
  P1DIR |= 0xFF;                        // P1.0 output
  P1OUT &= ~0x01;                       // Clear P1.0
                                        // unused ports to output
  P2DIR =  0xFF;
  P3DIR =  0xFF;
  P4DIR =  0xFF;
  P5DIR =  0xFF;
  P6DIR =  0xFF;
  U0CTL |= I2C + SYNC;                  // Recommended init procedure
  U0CTL &= ~I2CEN;                      // Recommended init procedure
  I2CTCTL |= I2CSSEL1;                  // SMCLK
  I2CNDAT = 0x02;                       // Read two bytes
  I2CSA = 0x0048;                       // Slave Address is 048h
  I2CIE = RXRDYIE;                      // Enable RXRDYIFG interrupt
  U0CTL |= I2CEN;                       // Enable I2C
  while (1)
  {
    U0CTL |= MST;                       // Master mode
    I2CTCTL |= I2CSTT + I2CSTP;         // Initiate transfer
    _BIS_SR(CPUOFF+GIE);                // Enter LPM0, Enable interrupts
  } 

}
// CRC-8 Algorithm to check the PEC
unsigned char crc8MakeBitwise(unsigned char crc, unsigned char poly, unsigned char *pmsg, unsigned int msg_size)
{
    volatile unsigned int i, j;
    unsigned char mg,carry = 0;

    crc ^= *pmsg++;

    for(i = 0 ; i < (msg_size-1) ; i ++)
    {
        mg = *pmsg++;

        for(j = 0 ; j < 8 ; j++)
        {
         carry = (crc & 0x80);
         crc = (crc << 1) | (mg >> 7);
         if(carry) crc ^= poly;
         mg <<= 1;
        }
    }
   
    return(crc);
}

// Common ISR for I2C Module
#pragma vector=USART0TX_VECTOR
__interrupt void I2C_ISR(void)
{
 switch(I2CIV)
 {
   static unsigned int count = 0;       // Counter to determine Byte#
   case  0: break;                      // No interrupt
   case  2: break;                      // Arbitration lost
   case  4: break;                      // No Acknowledge
   case  6: break;                      // Own Address
   case  8: break;                      // Register Access Ready
   case 10:                             // Receive Ready
     
     if (count == 0)
     {
      RXData = I2CDRB;                   // RX data
      count ++;
      msg[0] = I2CSA;
      msg[1] = RXData;
     }
     else
     {
      TestData = I2CDRB;
      msg[2] = TestData;
      count = 0;
      crc8 = crc8MakeBitwise(CRC8_INIT_REM, CRC8_POLY, msg, CRC_MSG_SIZE);
      _BIC_SR_IRQ(CPUOFF);               // Clear LPM0
     }
     if (crc8 != 0)
     P1OUT = 0x01;
     else
     P1OUT = 0;
     break; 
   case 12: break;                       // Transmit Ready
   case 14: break;                       // General Call
   case 16: break;                       // Start Condition
 }
}

  